/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */  

/**
 * \file    ddr3_test.c
 *
 * \brief   Support routines for DDR test application
 *
 * This file contains the support function for DDR Test application.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "stdio.h"
#include "dm388_EVM.h"
#include "ddr3.h"

//#define DDR3_DEBUG 1
/*
 *====================
 * Static declarations
 *====================
 */
static UINT8 tx[DDR_RW_SIZE];
static UINT8 rx[DDR_RW_SIZE];

 /*
 *====================
 * Function implementations
 *====================
 */


INT32 ddr3_pattern_test
(
	UINT32* pu32BaseAddr,
	UINT32 u32Size,
	UINT32 u32Pattern
)
{
	INT32 u32RetVal = SUCCESS;
	UINT32 u32Count = 0;
	UINT32* pu32Addr = pu32BaseAddr;

	/* ddr3 on EMIF 0 */
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		*pu32Addr++ = u32Pattern;
	}

	/* Verify the pattern written to memory */
	pu32Addr = pu32BaseAddr;
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		if (*pu32Addr++ != u32Pattern)
		{
			platform_write ("Memory cell verification failed at addr 0x%x.\r\n",
					pu32Addr);
			u32RetVal = FAILED;
			break;
		}
	}

	 return (u32RetVal);
}

INT32 ddr3_random_pattern_test ()
{
	UINT32* pu32BaseAddr;
	UINT32 u32Size;
	UINT32 u32Pattern;
    INT32 u32RetVal;
	UINT16 i;


	pu32BaseAddr =(UINT32*)DDR0_BASE;
	/* Write the known patterns to all banks of the DDR */
	for ( i = 0; i < DDR0_BANK_NUM; i++)
	{
		pu32BaseAddr += 100;
		u32Size = 0x100;
		u32Pattern = 0xAAAAAAAA;
#ifdef DDR3_DEBUG
		platform_write ("0x%xPattern test at location 0x%x of size 0x%x \r\n",u32Pattern,pu32BaseAddr,u32Size);
#endif
		u32RetVal = ddr3_pattern_test (pu32BaseAddr,u32Size,u32Pattern);
		if (u32RetVal != SUCCESS)
		{
			platform_write ("DDR3 Pattern Memory test failed.\r\n");
			u32RetVal = FAILED;
			break;
		}
		pu32BaseAddr += 100;
		u32Size = 0x100;
		u32Pattern = 0x55555555;
#ifdef DDR3_DEBUG
		platform_write ("0x%xPattern test at location 0x%x of size 0x%x \r\n",u32Pattern,pu32BaseAddr,u32Size);
#endif
		u32RetVal = ddr3_pattern_test (pu32BaseAddr,u32Size,u32Pattern);
		if (u32RetVal != SUCCESS)
		{
			platform_write ("DDR3 Pattern Memory test failed.\r\n");
			u32RetVal = FAILED;
			break;
		}
	}
    return u32RetVal;
}

INT32 ddr3IncDatatest
(
	UINT32* pu32BaseAddr, 
	UINT32 u32Size 
)
{
	INT32 u32RetVal = SUCCESS;
	UINT32 u32Count = 0;
	UINT32* pu32Addr = pu32BaseAddr;
	
	/* ddr3 on EMIF 0 */
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		*pu32Addr++ = u32Count;		
	}
	
	/* Verify the pattern written to memory */
	pu32Addr = pu32BaseAddr;
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		if (*pu32Addr++ != u32Count)
		{
			platform_write ("Incremental pattern Memory cell verification failed at addr 0x%x.\r\n",
					pu32Addr);
			u32RetVal = FAILED;
			break;
		}			
	}
	 
	 return (u32RetVal);
}

 /** **************************************************************************
 * \n \brief Routine to test ddr3 memory.
 *
 * This routine tests the ddr3 by writing into all banks and reading from it
 * then compares both .This routine takes  ddrctrl as parameter to select
 * the controller.
 * 
 * \param  ddrctrl   [IN]   To select controller
 *
 * \return
 * \n      return 0  for success  - Description
 * \n      return 1  for error   - Description
 */
INT16 ddr3_seq_rw_test (UINT32 ddr_rw_size)
{
    UINT32 i, j;
    UINT8* p8;
    UINT8 * src, *dst;
    UINT32 ddr_base, ddr_size,ddr_banks;
    ddr_base = DDR0_BASE;
	ddr_size = DDR0_SIZE;
	ddr_banks = DDR0_BANK_NUM;

    /* Write the known patterns to all banks of the DDR */
    for ( i = 0; i < ddr_banks; i++)
    {
        /* Create the test pattern */
        p8 = ( UINT8* )tx;
        for ( j = 0 ; j < ddr_rw_size ; j++ )
            *p8++ = ( UINT8 )( j + i );

		/* Write the size of DDR_RW_SIZE to the bank of ddr3 memory */
		src = tx;
		dst = (UINT8 *) (ddr_base + i * (ddr_size/ddr_banks)); /* 16 MB */
#ifdef DDR3_DEBUG
		platform_write ("writing data of size 0x%x to dst = 0x%x\r\n", ddr_rw_size,dst);
#endif
		ddr3_write ((UINT16 *)src, (UINT16 *)dst, ddr_rw_size);
    }

    /* Clear the receive buffer */
    for ( j = 0 ; j < ddr_rw_size ; j++ )
        rx[j] = 0;

    /* Read and verify ddr3 */
    for ( i = 0 ; i < ddr_banks ; i++ )
    {
        /* Read a ddr3 bank */
		src = (UINT8 *)(ddr_base + i * (ddr_size/ddr_banks));
		//dst = rx;
#ifdef DDR3_DEBUG
		platform_write ("Reading data of size 0x%x from src = 0x%x\r\n", ddr_rw_size,src);
#endif
		ddr3_read((UINT16 *)src, (UINT16 *)rx, ddr_rw_size);

        /* Check the pattern */
        p8 = ( UINT8* )rx;
        for ( j = 0 ; j < ddr_rw_size ; j++ )
            if ( ( *p8++ ) != ( UINT8 )( i + j ) )
                return 1;  // Fail
    }

    return 0;
}

 /** **************************************************************************
 * \n \brief Routine to test ddr3 memory.
 *
 * This routine tests the ddr3 by writing into all banks and reading from it
 * then compares both .This routine takes  ddrctrl as parameter to select
 * the controller.
 * 
 * \param  ddrctrl   [IN]   To select controller
 *
 * \return
 * \n      return 0  for success  - Description
 * \n      return 1  for error   - Description
 */
INT16 ddr3_rw_test ()
{
    INT16 i, j;
    UINT8* p8;
    UINT8 * src, *dst;
    UINT32 ddr_base, ddr_size, ddr_banks;

	ddr_base = DDR0_BASE;
	ddr_size = DDR0_SIZE;
	ddr_banks = DDR0_BANK_NUM;

    /* Write the known patterns to all banks of the DDR */
    for ( i = 0; i < ddr_banks; i++)
    {
        /* Create the test pattern */
        p8 = ( UINT8* )tx;
        for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
            *p8++ = ( UINT8 )( j + i );

		/* Write the size of DDR_RW_SIZE to the bank of ddr3 memory */
		src = tx;
		dst = (UINT8 *) (ddr_base + i * (ddr_size/ddr_banks)); /* 16 MB */
#ifdef DDR3_DEBUG
		platform_write ("dst = 0x%x\r\n", dst);
#endif
		ddr3_write ((UINT16 *)src, (UINT16 *)dst, DDR_RW_SIZE);
    }

    /* Clear the receive buffer */
    for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
        rx[j] = 0;

    /* Read and verify ddr3 */
    for ( i = 0 ; i < ddr_banks ; i++ )
    {
        /* Read a ddr3 bank */
		src = (UINT8 *)(ddr_base + i * (ddr_size/ddr_banks));
#ifdef DDR3_DEBUG
		platform_write ("src = 0x%x\r\n", src);
#endif
		//dst = rx;
        ddr3_read((UINT16 *)src, (UINT16 *)rx/*dst*/, DDR_RW_SIZE);

        /* Check the pattern */
        p8 = ( UINT8* )rx;
        for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
            if ( ( *p8++ ) != ( UINT8 )( i + j ) )
                return 1;  // Fail
    }

    return 0;
}

/** **************************************************************************
 * \n \brief Routine to test the ddr3 with cache_enable option
 *
 * This routine takes cache_enable option as a parameter and does DDR read 
 * write tests.
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
 
INT32 ddr3_test (void *testargs)
{
	INT32 u32RetVal = SUCCESS;
	UINT32 ddr_size = 0x100000;

    platform_write("\n********************************\n\r");
    platform_write(  "            DDR3 Test           \n\r");
    platform_write(  "********************************\n\r");

#ifdef PRE_COMPLIANCE
    /* Initialize the DDR interface if any */
    //ddr3_init( );

    platform_write("\nRunning DDR3 Sequential Memory Read/Write Test\n");
    u32RetVal = ddr3_pattern_test ((UINT32*)DDR0_BASE, ddr_size, 0xAAAAAAAA);
	if (SUCCESS != u32RetVal)
	{
		platform_write ("DDR3 Sequential Memory Read/Write Test failed\r\n");
		u32RetVal = FAILED;
	}
	else
		platform_write ("DDR3 Sequential Memory Read/Write Test passed\r\n");

#else


	/* testing the pattern write/readback */
	platform_write ("Carrying out 0xAAAAAAAA pattern test for DDR[0].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR0_BASE, DDR0_SIZE, 0xAAAAAAAA);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Pattern 0xAAAAAAAA write Test failed for DDR0.\r\n");
	}
	platform_write ("Carrying out 0x55555555 pattern test for DDR[0].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR0_BASE, DDR0_SIZE, 0x55555555);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Pattern 0x55555555 write Test failed for DDR0.\r\n");
	}

	platform_write ("Carrying out 0xAAAAAAAA pattern test for DDR[1].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR1_BASE, DDR1_SIZE, 0xAAAAAAAA);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Pattern 0xAAAAAAAA write Test failed for DDR1.\r\n");
	}
	platform_write ("Carrying out 0x55555555 pattern test for DDR[1].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR1_BASE, DDR1_SIZE, 0x55555555);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Pattern 0x55555555 write Test failed for DDR1.\r\n");
	}


	platform_write ("Carrying out Incremental pattern test for DDR[0].\r\n");
	u32RetVal = ddr3IncDatatest ((UINT32*)DDR0_BASE, ddr_size/*DDR0_SIZE*/);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Incremental Pattern write Test failed for DDR[0].\r\n");
	}

	platform_write ("Carrying out Incremental pattern test for DDR[1].\r\n");
	u32RetVal = ddr3IncDatatest ((UINT32*)DDR1_BASE, DDR1_SIZE);
	if (SUCCESS != u32RetVal)
	{
		u32RetVal=FAILED;
		platform_write ("Incremental Pattern write Test failed for DDR[1].\r\n");
	}
#endif
	if (SUCCESS == u32RetVal)
	{
	platform_write("\nDDR3 Test Passed!\n");
	platform_write("\nDDR3 Tests Completed!!\n");
	platform_write("-----------x----------");

	}
	else
	{
	platform_write("\nDDR3 Test failed!\n");
	platform_write("\nDDR3 Tests Completed!!\n");
	platform_write("-----------x----------");
	}
return u32RetVal;

}
